package gov.va.vamf.scheduling.varutility.resource;

import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import gov.va.vamf.scheduling.varutility.domain.ClinicalServices;
import gov.va.vamf.scheduling.varutility.domain.DirectBookingEligibilityCriteria;
import gov.va.vamf.scheduling.varutility.linkbuilder.LinkBuilder;
import gov.va.vamf.scheduling.varutility.service.ClinicalServicesService;
import gov.va.vamf.scheduling.varutility.service.DirectBookingEligibilityCriteriaService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.stereotype.Component;

import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.UriInfo;
import java.security.Principal;
import java.util.Date;
import java.util.List;

@Path("/direct-booking-eligibility-criteria/site/{site-code}")
@Component
@Scope("request")
public class DirectBookingEligibilityCriteriaResource {

    @Autowired
    private DirectBookingEligibilityCriteriaService directBookingEligibilityCriteriaService;

    @Autowired
    private ClinicalServicesService clinicalServicesService;

    /**
     * Fetch Patient History options for Facility (Site).
     * @param uriInfo
     * @param siteCode
     * @return DirectBookingEligibilityCriteria
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public DirectBookingEligibilityCriteria fetchPatientHistoryOptions(
        @Context UriInfo uriInfo,
        @PathParam("site-code") String siteCode) {

        List<ClinicalServices> clinicalServices = clinicalServicesService.fetchForDirectScheduling();
        DirectBookingEligibilityCriteria criteria = directBookingEligibilityCriteriaService.fetchPatientHistoryOptions(siteCode, clinicalServices, uriInfo);
        return criteria;
    }

    /**
     * Save Patient History options for Facility (Site).
     * @param uriInfo
     * @param securityContext
     * @param siteCode
     * @return DirectBookingEligibilityCriteria
     */
    @POST
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Consumes({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    public DirectBookingEligibilityCriteria savePatientHistoryOptions(DirectBookingEligibilityCriteria criteria,
        @Context UriInfo uriInfo, @Context SecurityContext securityContext,
        @PathParam("site-code") String siteCode) {

        List<ClinicalServices> results = clinicalServicesService.fetchForDirectScheduling();
        criteria.validate(siteCode, results);
        criteria.setCreatedDate(new Date());
        criteria.setCreatedBy(getUserName(securityContext));
        criteria.cleanupDataForSave();
        DirectBookingEligibilityCriteria savedCriteria = directBookingEligibilityCriteriaService.savePatientHistoryOptions(criteria);
        LinkBuilder.fillLink(savedCriteria, uriInfo.getRequestUri());
        return savedCriteria;
    }

    /**
     * Update Patient History options for Facility (Site).
     * @param uriInfo
     * @param securityContext
     * @param siteCode
     * @param id
     * @return DirectBookingEligibilityCriteria
     */
    @PUT
    @Path( "/id/{id}" )
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Consumes({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    public DirectBookingEligibilityCriteria updatePatientHistoryOptions(DirectBookingEligibilityCriteria criteria,
        @Context UriInfo uriInfo, @Context SecurityContext securityContext,
        @PathParam("site-code") String siteCode,
        @PathParam("id") String id) {

        List<ClinicalServices> results = clinicalServicesService.fetchForDirectScheduling();
        criteria.validate(siteCode, results);
        criteria.setLastModifiedDate(new Date());
        criteria.setModifiedBy(getUserName(securityContext));
        criteria.cleanupDataForSave();
        DirectBookingEligibilityCriteria updatedCriteria = directBookingEligibilityCriteriaService.updatePatientHistoryOptions(criteria);
        LinkBuilder.fillLink(updatedCriteria, uriInfo.getRequestUri());
        return updatedCriteria;
    }

    private String getUserName(SecurityContext securityContext) {
        Principal principal = securityContext.getUserPrincipal();
        AppUser appUser = (AppUser)((OAuth2Authentication) principal).getUserAuthentication().getPrincipal();
        return appUser.getMhpUser().getDisplayName();
    }
}
